package cz.drg.clasificator.readers;

import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.util.HeaderList;
import cz.drg.clasificator.util.HeaderListReader;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import static cz.drg.clasificator.util.OutputHelper.*;
import java.util.zip.ZipFile;

public abstract class BaseReader implements InputReader, HeaderListReader{

    private String DELIMITER;
    private String ROW_SPLIT_REGEX;
    
    private final ZipFile pmmlInput; 
    private Map<String, Integer> headers;
    private List<List<String>> columns;
    private int numberOfLines = 0;
    
    private boolean alreadyReadFrom = false;

    public BaseReader(ZipFile pmmlInput, String delimiter) {
        this.pmmlInput = pmmlInput;
        
        DELIMITER = delimiter;
        ROW_SPLIT_REGEX = DELIMITER+"(?=(?:[^\"]*\"[^\"]*\")*[^\"]*$)";
    }
    
    protected void init(){
        
        headers = new HashMap<>();
        columns = new ArrayList<>();
        
    }

    protected abstract List<String> getData();
    
    protected void readSource(List<String> lines){
        
        columns.clear();
        
        numberOfLines = lines.size();
        
        if(!alreadyReadFrom){
            numberOfLines = lines.size() - 1;
        }
        
        for (int rowIndex = 0; rowIndex < lines.size(); rowIndex++) {
            String[] rowColumns = lines.get(rowIndex).split(ROW_SPLIT_REGEX, -1);
            
            for (int columnIndex = 0; columnIndex < rowColumns.length; columnIndex++) {
                String column = rowColumns[columnIndex].replaceAll("\"", "");
                
                //if its header add the whole line to header column map
                //since addition of processing of file in entry batches, its needed
                //to store header line from only first batch of input
                if(rowIndex == 0 && !alreadyReadFrom){
                    headers.put(column, columnIndex);
                }
                else{
                    
                    //check if the row has same ammount of collumns as header
                    if(rowColumns.length != headers.size()){
                        numberOfLines--;
                        break;
                    }
                    
                    //initialize column lists if empty
                    if(columns.size() <= columnIndex){
                        columns.add(new ArrayList<>());
                    }
                    
                    columns.get(columnIndex).add(column);
                }
            }
        }
        
        alreadyReadFrom = true;
    }

    @Override
    public Map<String, Integer> getHeadersWithIndexes() {
        return headers;
    }

    @Override
    public List<List<String>> getColumnLists() {
        return columns;
    }

    @Override
    public int getNumberOfLines() {
        return numberOfLines;
    }

    @Override
    public ZipFile readPmmlInput() {
        return pmmlInput;
    }

    @Override
    public HeaderList readNextEntryBatch() {
        
        readSource(getData());
        return new HeaderList(this);
        
    }
    
    @Override
    public void initialize() {}

    @Override
    public void close() {}
    
}
